package theorycrafter.fitting

import kotlin.math.min


/**
 * Specifies the amount of spool cycles to set on a module.
 */
sealed interface SpoolupCycles {


    /**
     * Specifies that the maximum number of spool cycles a module supports should be set.
     */
    data object Maximum: SpoolupCycles


    /**
     * Specifies an exact number of spool cycles to be set.
     */
    class Number(val value: Double): SpoolupCycles


    /**
     * Returns the actual number of spool cycles to be set on the given module.
     */
    fun valueFor(module: Module): Double {
        val maxSpoolupCycles = module.maxSpoolupCycles ?: 0.0
        return when (this) {
            is Maximum -> maxSpoolupCycles
            is Number -> min(value, maxSpoolupCycles)
        }
    }


    companion object {


        /**
         * Returns the [SpoolupCycles] of the given module.
         */
        fun of(module: Module): SpoolupCycles? {
            val count = module.spoolupCycles?.value ?: return null
            val maxCount = module.maxSpoolupCycles!!
            return if (count == maxCount) Maximum else Number(count)
        }


    }


}
