package theorycrafter.formats

import eve.data.ModuleSlotType
import eve.data.ModuleType
import theorycrafter.TheorycrafterContext
import theorycrafter.fitting.Module
import theorycrafter.storage.ExtraAttribute
import theorycrafter.storage.SpoolupCyclesExtraAttribute
import theorycrafter.storage.StoredFit
import theorycrafter.utils.defaultInitialState


/**
 * Returns the default state for a module when importing it.
 */
fun ModuleType.importedState(isOffline: Boolean = false): Module.State = when {
    slotType == ModuleSlotType.RIG -> Module.State.ONLINE  // Rigs' state is always "online"; they use "enabled" instead
    isOffline -> Module.State.OFFLINE
    else -> defaultInitialState()
}


/**
 * Returns the list if [ExtraAttribute]s to set on a module when importing it.
 */
fun ModuleType.importedExtraAttributes(): List<ExtraAttribute<Module>> {
    val eveData = TheorycrafterContext.eveData
    return if (hasAttribute(eveData.attributes.spoolupCycles))
        listOf(SpoolupCyclesExtraAttribute.MAX)
    else
        emptyList()
}


/**
 * A command, friendly or hostile remote fit.
 */
class ImportedRemoteFitEffect(
    /**
     * The affecting fit.
     */
    val affectingFit: StoredFit,

    /**
     * The affected fit.
     */
    val affectedFit: StoredFit,

    /**
     * The amount of the effect.
     */
    val amount: Int,

    /**
     * Whether the effect is enabled.
     */
    val enabled: Boolean
)


/**
 * Information about a fit that could not be loaded because something in it was wrong.
 */
class UnloadedFit(
    /**
     * The name of the fit.
     */
    val name: String,

    /**
     * The name of ship type.
     */
    val shipTypeName: String
)


/**
 * The result of parsing fits.
 */
class FitsImportResult(
    /**
     * The total number of fits in the input.
     */
    val totalFitsCount: Int,

    /**
     * The fits that have been loaded successfully.
     */
    val fits: List<StoredFit>,

    /**
     * Command fits.
     */
    val commandFitEffects: List<ImportedRemoteFitEffect> = emptyList(),

    /**
     * Friendly effect fits.
     */
    val friendlyFitEffects: List<ImportedRemoteFitEffect> = emptyList(),

    /**
     * Hostile effect fits.
     */
    val hostileFitEffects: List<ImportedRemoteFitEffect> = emptyList(),

    /**
     * The fits that could not be loaded.
     */
    val unsuccessful: List<UnloadedFit>
)
