package theorycrafter.ui.tournaments

import androidx.compose.foundation.background
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.LocalContentColor
import androidx.compose.material.LocalTextStyle
import androidx.compose.material.Text
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.key.Key
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.withStyle
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.DpRect
import androidx.compose.ui.unit.LayoutDirection
import androidx.compose.ui.unit.dp
import compose.input.*
import compose.utils.LocalKeyShortcutsManager
import compose.utils.VSpacer
import compose.widgets.*
import eve.data.ShipType
import eve.data.asDps
import eve.data.asHitPoints
import eve.data.asHitpointsPerSecond
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.withContext
import theorycrafter.FitHandle
import theorycrafter.TestTags
import theorycrafter.TheorycrafterContext
import theorycrafter.fitting.Fit
import theorycrafter.fitting.ItemDefense
import theorycrafter.tournaments.*
import theorycrafter.tournaments.Composition
import theorycrafter.ui.*
import theorycrafter.ui.fiteditor.*
import theorycrafter.ui.widgets.HORIZONTAL_SLOT_ROW_PADDING
import theorycrafter.ui.widgets.SLOT_ROW_PADDING
import theorycrafter.ui.widgets.Selector
import theorycrafter.ui.widgets.SlotRow
import theorycrafter.utils.AutoSuggest
import theorycrafter.utils.onEmptyQueryReturn
import theorycrafter.utils.shipTypeTraitsTooltip
import theorycrafter.utils.thenIf


/**
 * The grid column indices.
 */
private object GridCols {
    const val STATE_ICON = 0
    const val SHIP_ICON = 1
    const val SHIP = 2
    const val COST = 3
    const val FIT = 4
    const val EHP = 5
    const val ACTIVE_TANK = 6
    const val DPS = 7
    const val COUNT = 8
}


/**
 * The alignment in each column.
 */
private val ColumnAlignment = listOf(
    Alignment.Center,       // state icon
    Alignment.Center,       // ship icon
    Alignment.CenterStart,  // ship
    Alignment.CenterEnd,    // cost
    Alignment.CenterStart,  // fit
    Alignment.CenterEnd,    // ehp
    Alignment.CenterEnd,    // tank
    Alignment.CenterEnd,    // dps
)


/**
 * The widths of the columns.
 */
private val ColumnWidths = listOf(
    24.dp,           // state icon
    TheorycrafterTheme.sizes.eveTypeIconSmall + 6.dp,           // ship icon
    160.dp,          // ship type
    80.dp,           // points
    Dp.Unspecified,  // fit name
    70.dp,           // EHP
    100.dp,          // Active Tank
    80.dp,           // DPS
)


/**
 * The padding of the "Cost" column.
 *
 * This is needed because otherwise it's too close to the fit column.
 */
private val COST_COLUMN_PADDING = PaddingValues(end = TheorycrafterTheme.spacing.xxlarge)


/**
 * The padding modifier of the "Cost" column.
 */
private val COST_COLUMN_PADDING_MODIFIER = Modifier.padding(COST_COLUMN_PADDING)


/**
 * The grid header row.
 */
@Composable
private fun HeaderRow(
    tournament: Tournament,
    modifier: Modifier = Modifier
) {
    SimpleGridHeaderRow(
        modifier = modifier
            .padding(HORIZONTAL_SLOT_ROW_PADDING),
        columnWidths = ColumnWidths,
        defaultCellContentAlignment = ColumnAlignment::get
    ) {
        CompositionLocalProvider(LocalTextStyle provides TextStyle(fontWeight = FontWeight.Bold)) {
            EmptyCell(index = GridCols.STATE_ICON)
            EmptyCell(index = GridCols.SHIP_ICON)
            TextCell(index = GridCols.SHIP, "Ship")
            TextCell(
                index = GridCols.COST,
                text = when (tournament.rules.compositionRules) {
                    is PointsCompositionRules -> "Cost"
                    is DraftCompositionRules -> "Points"
                },
                modifier = COST_COLUMN_PADDING_MODIFIER
            )
            TextCell(index = GridCols.FIT, "Fit")
            TextCell(index = GridCols.EHP, "EHP")
            TextCell(index = GridCols.ACTIVE_TANK, "Active Tank")
            TextCell(index = GridCols.DPS, "DPS")
        }
    }
}


/**
 * Computes the number of row slots to display for the given composition.
 */
private val Composition.visibleSlotCount: Int
    get() = when(val rules = tournament.rules.compositionRules) {
        is PointsCompositionRules -> rules.maxCompositionSize.coerceAtLeast(size + 1)  // +1 for the empty slot
        is DraftCompositionRules -> rules.positions.size.coerceAtLeast(size) + 1  // +1 for replacement ship slot
    }


/**
 * The panel showing the composition and allows editing it.
 */
@Composable
fun CompositionEditor(
    composition: Composition,
    modifier: Modifier
) {
    val selectionModel = remember(composition) {
        CompositionShipsSelectionModel(composition)
    }
    // Update the selected index when the list of slots no longer includes it
    LaunchedEffect(selectionModel, selectionModel.maxSelectableIndex) {
        selectionModel.selectedIndex?.let {
            if (it > selectionModel.maxSelectableIndex)
                selectionModel.selectLast()
        }
    }
    val rules = composition.tournament.rules.compositionRules
    val shipsCost by remember(rules, composition) {
        derivedStateOf {
            rules.shipsCost(composition)
        }
    }
    val shipsIllegalityReason by remember(rules, composition) {
        derivedStateOf {
            rules.compositionShipsIllegalityReason(composition)
        }
    }

    val compositionExporter = rememberCompositionExporter()
    LocalKeyShortcutsManager.current.register(
        shortcut = CopyCompositionToClipboardKeyShortcut,
        action = { compositionExporter.copyToClipboard(composition) }
    )
    ContentWithScrollbar(modifier.testTag(TestTags.CompEditor.RootNode)) {
        Column(Modifier.verticalScroll(scrollState)) {
            HeaderRow(
                tournament = composition.tournament,
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(
                        top = TheorycrafterTheme.spacing.larger,
                        bottom = TheorycrafterTheme.spacing.xsmall
                    )
            )
            SimpleGrid(
                columnWidths = ColumnWidths,
                rowSelectionModel = selectionModel,
                defaultCellContentAlignment = ColumnAlignment::get,
                modifier = Modifier
                    .moveSelectionWithKeys(selectionModel),
            ) {
                val slotCount = composition.visibleSlotCount
                for (index in 0 until slotCount) {
                    val slotActions = remember(composition, index, selectionModel) {
                        ShipSlotActions(composition, index, selectionModel)
                    }

                    val ship = composition[index]
                    inRow(index) {
                        if (ship != null) {
                            ShipSlotRow(
                                composition = composition,
                                shipIndex = index,
                                ship = ship,
                                shipCost = shipsCost[index],
                                shipIllegalReason = shipsIllegalityReason[index],
                                selectionModel = selectionModel,
                                actions = slotActions,
                            )
                        }
                        else {
                            EmptyShipSlotRow(
                                composition = composition,
                                shipIndex = index,
                                selectionModel = selectionModel,
                                actions = slotActions,
                            )
                        }
                    }
                }
            }

            // The separator line
            Box(
                modifier = Modifier
                    .padding(
                        top = TheorycrafterTheme.spacing.large,
                        bottom = TheorycrafterTheme.spacing.small)
                    .padding(HORIZONTAL_SLOT_ROW_PADDING)
                    .background(LocalContentColor.current.copy(alpha = 0.4f))
                    .fillMaxWidth()
                    .height(1.dp)
            )

            SummationRow(
                composition = composition,
                shipsCost = shipsCost,
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(HORIZONTAL_SLOT_ROW_PADDING)
            )

            VSpacer(TheorycrafterTheme.spacing.larger)

            val sectionsModifier = Modifier
                .fillMaxWidth()
                .padding(horizontal = TheorycrafterTheme.spacing.horizontalEdgeMargin)

            UtilitySummary(
                composition = composition,
                modifier = sectionsModifier
            )

            VSpacer(TheorycrafterTheme.spacing.larger)

            CompositionNotes(
                composition = composition,
                modifier = sectionsModifier
            )
        }
    }
}


/**
 * Moves the slot selection when Up/Down/Home/End keys are pressed.
 */
@Composable
private fun Modifier.moveSelectionWithKeys(
    selectionModel: CompositionShipsSelectionModel,
): Modifier = with(FitEditorKeyShortcuts) {
    this@moveSelectionWithKeys
        .onKeyShortcut(SelectPrevRow) {
            selectionModel.selectPrevious()
        }
        .onKeyShortcut(SelectNextRow) {
            selectionModel.selectNext()
        }
        .onKeyShortcut(SelectionFirst) {
            selectionModel.selectFirst()
        }
        .onKeyShortcut(SelectionLast) {
            selectionModel.selectLast()
        }
}


/**
 * Groups actions on a composition ship slot.
 */
private class ShipSlotActions(
    private val composition: Composition,
    private val index: Int,
    private val selectionModel: CompositionShipsSelectionModel,
) {


    /**
     * Sets the [Composition.Ship] in the slot to a new one, with the given ship type.
     */
    fun setShip(shipType: ShipType) {
        val active = when (val rules = composition.tournament.rules.compositionRules) {
            is PointsCompositionRules -> true
            is DraftCompositionRules -> index in rules.positions.indices
        }
        composition[index] = Composition.Ship(composition, shipType, fitId = null, active = active)
    }


    /**
     * Removes the current ship from the slot.
     */
    fun clearSlot() {
        when (val rules = composition.tournament.rules.compositionRules) {
            is PointsCompositionRules -> composition[index] = null
            is DraftCompositionRules ->
                if (index in rules.positions.indices)
                    composition[index] = null
                else
                    composition.removeAt(index)

        }
    }


    /**
     * Inserts an empty slot at this index.
     */
    fun insertEmptySlot() {
        composition.insert(index, null)
    }


    /**
     * Removes the slot itself.
     */
    fun removeSlot() {
        composition.removeAt(index)
    }


    /**
     * Returns whether the ship can be moved up one slot.
     */
    fun canMoveUp() = index > 0


    /**
     * Moves the ship up one slot.
     */
    fun moveUp() {
        composition.swap(index, index - 1)
        selectionModel.selectIndex(index - 1)
    }


    /**
     * Returns whether the ship can be moved down one slot.
     */
    fun canMoveDown() = true  // We can always expand the composition


    /**
     * Moves the ship down one slot.
     */
    fun moveDown() {
        composition.swap(index, index + 1)
        selectionModel.selectIndex(index + 1)
    }


    /**
     * Moves the ship to the given slot.
     */
    fun moveTo(targetIndex: Int) {
        composition.move(fromIndex = index, toIndex = targetIndex)
        selectionModel.selectIndex(targetIndex)
    }


    /**
     * Adds another ship like the one in this slot.
     */
    fun addAnother() {
        val thisShip = composition[index] ?: return
        composition.insert(index+1, thisShip.copy())
    }


    /**
     * Removes another ship like the one in this slot.
     */
    fun removeOne() {
        val targetIndex = removeOneSlotIndex() ?: return
        if (targetIndex < index)
            composition[targetIndex] = null
        else
            composition.removeAt(targetIndex)
    }


    /**
     * Returns the index of the slot that will be removed by [removeOne]; `null` if none.
     */
    private fun removeOneSlotIndex(): Int? {
        val thisShip = composition[index] ?: return null
        fun isSameShipAsAt(index: Int): Boolean {
            val s = composition[index]
            return (s != null) && (s.shipType == thisShip.shipType) && (s.fitId == thisShip.fitId)
        }

        return (composition.size-1 downTo index+1).firstOrNull(::isSameShipAsAt) ?:
            (0 until index).firstOrNull(::isSameShipAsAt)
    }


    /**
     * Returns whether the composition has another ship like the one in this slot.
     */
    fun canRemoveOne(): Boolean {
        return removeOneSlotIndex() != null
    }


    /**
     * Toggles the active state of the ship.
     */
    fun toggleActive() {
        val ship = composition[index] ?: return
        val active = !ship.active
        ship.active = active

        // Enable/disable this ship's effect on all other ships in the composition
        val fitContext = TheorycrafterContext.fits
        val fitHandle = ship.fitId?.let { fitContext.handleById(it) } ?: return
        runBlocking {
            val fit = fitContext.engineFitOf(fitHandle)
            val targetFits = composition.ships.mapNotNull {  targetShip ->
                if ((targetShip == null) || (targetShip == ship))
                    return@mapNotNull null
                val targetFitHandle = targetShip.fitId?.let { fitContext.handleById(it) } ?: return@mapNotNull null
                fitContext.engineFitOf(targetFitHandle)
            }

            fitContext.modifyAndSave {
                for (targetFit in targetFits) {
                    val commandEffect = targetFit.commandEffects.find { it.source == fit }
                    val friendlyEffect = targetFit.friendlyEffects.find { it.source == fit }

                    if ((commandEffect != null) || (friendlyEffect != null)) {
                        commandEffect?.setEnabled(active)
                        friendlyEffect?.setEnabled(active)
                    }
                }
            }
        }
    }


}


/**
 * The shortcut to start editing the fit associated with a composition ship.
 */
private val EditFitKeyShortcut = KeyShortcut.anyOf(
    keys = listOf(Key.Enter, Key.NumPadEnter, Key.F2),
    keyModifier = KeyboardModifierMatcher.Alt
)


/**
 * The shortcut for inserting an empty ship slot.
 */
private val InsertEmptySlotKeyShortcut = KeyShortcut.anyOf(
    keys = listOf(Key.Plus, Key.NumPadAdd, Key.Equals),  // Allow = to match the + key on the regular numbers row
    keyModifier = KeyboardModifierMatcher.Command
)


/**
 * The shortcut for removing a ship slot.
 */
private val RemoveSlotKeyShortcuts = FitEditorKeyShortcuts.ClearSlot +
        KeyShortcut.anyOf(listOf(Key.Minus, Key.NumPadSubtract)) +
        KeyShortcut.anyOf(listOf(Key.Minus, Key.NumPadSubtract), KeyboardModifierMatcher.Command)


/**
 * The slot context action to insert an empty ship slot.
 */
private fun SlotContextAction.Companion.insertEmptySlot(actions: ShipSlotActions) =
    SlotContextAction(
        displayName = "Insert Empty Slot",
        icon = { Icons.Add() },
        shortcuts = InsertEmptySlotKeyShortcut,
        action = actions::insertEmptySlot
    )


/**
 * The slot context action to remove a ship slot.
 */
private fun SlotContextAction.Companion.removeEmptySlot(actions: ShipSlotActions) =
    SlotContextAction(
        displayName = "Remove Slot",
        icon = { Icons.Remove() },
        shortcuts = RemoveSlotKeyShortcuts,
        action = actions::removeSlot
    )


/**
 * Returns the [FitHandle] associated with the given composition ship.
 */
@Composable
private fun Composition.Ship.fitHandle(): FitHandle? {
    return remember(fitId, TheorycrafterContext.fits.handlesKey) {
        fitId?.let {
            TheorycrafterContext.fits.handleById(it)
        }
    }
}


/**
 * Returns whether in the given tournament, there are no dedicated ship slots.
 */
private fun Tournament.isFreeSlotTournament() = rules.compositionRules !is DraftCompositionRules


/**
 * The row displaying a non-empty ship slot.
 */
@Composable
private fun GridScope.ShipSlotRow(
    composition: Composition,
    shipIndex: Int,
    ship: Composition.Ship,
    shipCost: Int,
    shipIllegalReason: String?,
    selectionModel: SingleItemSelectionModel,
    actions: ShipSlotActions
) {
    val fitOpener = LocalFitOpener.current
    val fitHandle = ship.fitHandle()
    val canRemoveOne by remember(actions) {
        derivedStateOf {
            actions.canRemoveOne()
        }
    }
    val freeSlotTournament = composition.tournament.isFreeSlotTournament()
    val contextActions = remember(actions, fitOpener, fitHandle, canRemoveOne) {
        buildList {
            add(SlotContextAction.toggleEnabledState(actions::toggleActive))
            add(SlotContextAction.openFit(fitOpener, fitHandle))
            add(SlotContextAction.Separator)
            if (freeSlotTournament) {
                add(SlotContextAction.insertEmptySlot(actions))
                add(SlotContextAction.moveSlotUp(enabled = actions.canMoveUp(), action = actions::moveUp))
                add(SlotContextAction.moveSlotDown(enabled = actions.canMoveDown(), action = actions::moveDown))
                add(SlotContextAction.Separator)
            }
            add(SlotContextAction.clear(actions::clearSlot))
            if (freeSlotTournament) {
                add(SlotContextAction.addOneItem(actions::addAnother))
                add(
                    SlotContextAction.removeOneItem(
                        removeOne = actions::removeOne,
                        enabled = canRemoveOne,
                        showInContextMenu = true
                    )
                )
            }
        }
    }

    val isEditingState = remember { mutableStateOf(false) }
    var isEditingFit by remember { mutableStateOf(false) }

    fun startEditingFit() {
        isEditingFit = true
        isEditingState.value = true
    }

    SlotRow(
        modifier = Modifier
            .thenIf(!isEditingState.value) {
                onMousePress(clickCount = ClickCount.DOUBLE, keyboardModifier = KeyboardModifierMatcher.Alt) {
                    startEditingFit()
                }
                .onKeyShortcut(EditFitKeyShortcut) {
                    startEditingFit()
                }
            },
        modifierWhenNotEditing = Modifier
            .thenIf(freeSlotTournament) {
                dragShipToReorder(ship, actions)
            },
        contextActions = contextActions,
        invalidityReason = shipIllegalReason,
        isEditingState = isEditingState,
        editedRowContent = { onEditingCompleted ->
            if (isEditingFit) {
                EditedFitRowContent(
                    ship = ship,
                    shipCost = shipCost,
                    selectionModel = selectionModel,
                    onFitSelected = {
                        ship.fitId = it?.fitId
                        isEditingFit = false
                        onEditingCompleted()
                    },
                    onEditingCancelled = {
                        isEditingFit = false
                        onEditingCompleted()
                    }
                )
            }
            else {
                EditedShipTypeRowContent(
                    composition = composition,
                    shipIndex = shipIndex,
                    selectionModel = selectionModel,
                    currentShipType = ship.shipType,
                    onShipSelected = {
                        actions.setShip(it)
                        onEditingCompleted()
                    },
                    onEditingCancelled = onEditingCompleted
                )
            }
        },
    ) {
        ShipSlotContent(
            ship = ship,
            shipIndex = shipIndex,
            tournament = composition.tournament,
            shipCost = shipCost,
            actions = actions,
            onFitDoubleClicked = {
                startEditingFit()
            }
        )
    }
}


/**
 * Returns a modifier for dragging the given ship to place it at a new slot.
 */
context(GridScope)
@Composable
private fun Modifier.dragShipToReorder(
    ship: Composition.Ship,
    actions: ShipSlotActions
): Modifier {

    return this.dragRowToReorder(
        draggableContent = { DraggedShipSlotRepresentation(ship) },
        canMoveToRow = { it <= ship.composition.visibleSlotCount },
        onDrop = { draggedRowIndex, dropRowIndex ->
            val targetRowIndex = if (dropRowIndex > draggedRowIndex) dropRowIndex-1 else dropRowIndex
            actions.moveTo(targetRowIndex)
        }
    )
}


/**
 * The dragged module slot.
 */
@Composable
private fun GridScope.DraggedShipSlotRepresentation(ship: Composition.Ship) {
    draggedRow(
        rowIndex = 0,
        modifier = Modifier
            .fillMaxWidth()
            .background(TheorycrafterTheme.colors.draggedSlotBackground())
            .padding(SLOT_ROW_PADDING)
            .height(TheorycrafterTheme.sizes.fitEditorSlotRowHeight)  // Make the height independent of editing
    ) {
        emptyCell(GridCols.STATE_ICON)
        emptyCell(GridCols.SHIP_ICON)
        cell(GridCols.SHIP, colSpan = GridCols.COUNT - GridCols.SHIP, contentAlignment = Alignment.CenterStart) {
            Text(ship.shipType.name)
        }
    }
}


/**
 * The content of a non-empty composition ship slot.
 */
@Composable
private fun GridScope.GridRowScope.ShipSlotContent(
    ship: Composition.Ship,
    shipIndex: Int,
    tournament: Tournament,
    shipCost: Int,
    actions: ShipSlotActions,
    onFitDoubleClicked: () -> Unit,
) {
    val fitHandle = ship.fitHandle()
    val fit by produceState<Fit?>(null, fitHandle) {
        value = fitHandle?.let { TheorycrafterContext.fits.engineFitOf(it) }
    }

    cell(GridCols.STATE_ICON, modifier = Modifier) {
        Icons.ItemEnabledState(
            enabled = ship.active,
            modifier = Modifier
                .onMousePress(consumeEvent = true) {  // Consume to prevent selecting the row
                    actions.toggleActive()
                }
                .onMousePress(MouseButton.Middle, consumeEvent = true) {  // Consume just in case
                    actions.toggleActive()
                }
        )
    }

    ShipIconCell(ship)

    cell(
        cellIndex = GridCols.SHIP,
        modifier = defaultCellModifier
            .shipTypeTraitsTooltip(
                shipType = ship.shipType,
                delayMillis = 1000
            )
    ) {
        val position = (tournament.rules.compositionRules as? DraftCompositionRules)?.positions?.getOrNull(shipIndex)
        if (position == null) {
            SingleLineText(ship.shipType.shortName())
        } else {
            val lightStyle = LocalTextStyle.current.copy(fontWeight = FontWeight.ExtraLight)
            SingleLineText(
                text = buildAnnotatedString {
                    withStyle(lightStyle.toSpanStyle()) {
                        append(position.prefix)
                        append(" ")
                    }
                    append(ship.shipType.shortName())
                }
            )
        }
    }

    ShipCost(ship, shipCost)

    cell(
        cellIndex = GridCols.FIT,
        modifier = Modifier
            .onMousePress(clickCount = ClickCount.DOUBLE, consumeEvent = true) {
                onFitDoubleClicked()
            }
    ) {
        SingleLineText(
            text = fitHandle?.name ?: "None",
            fontWeight = if (fitHandle == null) FontWeight.ExtraLight else null,
        )
    }

    cell(GridCols.EHP, contentAlignment = Alignment.CenterEnd) {
        if (ship.active) {
            fit?.let { fit ->
                SingleLineText(fit.defenses.ehp.asHitPoints(ehp = true, withUnits = false))
            }
        }
    }

    cell(GridCols.ACTIVE_TANK) {
        if (ship.active) {
            fit?.let { fit ->
                with(fit.defenses) {
                    fun ItemDefense.hasRepairs() = (hpRepairedLocally > 0) || (hpRepairedRemotely > 0)
                    val includeShieldRegen = shield.hasRepairs() || (!armor.hasRepairs() && !structure.hasRepairs())
                    val totalReps = fit.defenses.ehpRepairedLocally +
                            fit.defenses.ehpRepairedByRemoteEffects +
                            (if (includeShieldRegen) shield.peakRegenEhpPerSecond else 0.0)
                    SingleLineText(totalReps.asHitpointsPerSecond(ehp = true, withUnits = false))
                }
            }
        }
    }

    cell(GridCols.DPS, contentAlignment = Alignment.CenterEnd) {
        if (ship.active) {
            fit?.let {
                SingleLineText(it.firepower.totalDps.asDps(withUnits = false))
            }
        }
    }
}


/**
 * A cell showing the ship's icon.
 */
@Composable
private fun GridScope.GridRowScope.ShipIconCell(ship: Composition.Ship) {
    cell(GridCols.SHIP_ICON) {
        Icons.EveItemType(
            itemType = ship.shipType,
            modifier = Modifier
                .size(TheorycrafterTheme.sizes.eveTypeIconSmall)
                .padding(2.dp)
        )
    }
}


/**
 * A cell showing the ship's cost.
 */
@Composable
private fun GridScope.GridRowScope.ShipCost(
    ship: Composition.Ship,
    shipCost: Int,
) {
    cell(GridCols.COST, modifier = defaultCellModifier.then(COST_COLUMN_PADDING_MODIFIER)) {
        if (ship.active) {
            SingleLineText(text = "$shipCost")
        }
    }
}


/**
 * The content of the row when the ship type is being edited.
 */
@Composable
private fun GridScope.GridRowScope.EditedShipTypeRowContent(
    composition: Composition,
    shipIndex: Int,
    selectionModel: SingleItemSelectionModel,
    currentShipType: ShipType?,
    onShipSelected: (ShipType) -> Unit,
    onEditingCancelled: () -> Unit
) {
    emptyCell(cellIndex = GridCols.STATE_ICON)  // No state icon
    emptyCell(cellIndex = GridCols.SHIP_ICON)  // No ship icon
    cell(
        cellIndex = GridCols.SHIP,
        colSpan = GridCols.COST - GridCols.SHIP + 1
    ) {
        when (val compRules = composition.tournament.rules.compositionRules) {
            is PointsCompositionRules ->
                PointsBasedShipTypeSelector(
                    composition = composition,
                    rules = compRules,
                    selectionModel = selectionModel,
                    currentShipType = currentShipType,
                    onShipTypeSelected = onShipSelected,
                    onEditingCancelled = onEditingCancelled
                )
            is DraftCompositionRules ->
                PositionBasedShipTypeSelector(
                    shipIndex = shipIndex,
                    rules = compRules,
                    selectionModel = selectionModel,
                    onShipTypeSelected = onShipSelected,
                    onEditingCancelled = onEditingCancelled
                )
        }
    }
}


/**
 * The content of the row when the fit is being edited.
 */
@Composable
private fun GridScope.GridRowScope.EditedFitRowContent(
    ship: Composition.Ship,
    shipCost: Int,
    selectionModel: SingleItemSelectionModel,
    onFitSelected: (FitHandle?) -> Unit,
    onEditingCancelled: () -> Unit
) {
    emptyCell(cellIndex = GridCols.STATE_ICON)  // No state icon
    ShipIconCell(ship)
    cell(cellIndex = GridCols.SHIP) {
        Text(ship.shipType.shortName())
    }
    ShipCost(ship, shipCost)
    val fitSelectorColSpan = 3
    cell(
        cellIndex = GridCols.FIT,
        colSpan = fitSelectorColSpan
    ) {
        FitSelector(
            shipType = ship.shipType,
            selectionModel = selectionModel,
            onFitSelected = onFitSelected,
            onEditingCancelled = onEditingCancelled
        )
    }

    // This is needed to limit the width of the fit selector
    val restColIndex = GridCols.FIT + fitSelectorColSpan
    emptyCell(
        cellIndex = restColIndex,
        colSpan = GridCols.COUNT - restColIndex
    )
}


/**
 * A ship type selection widget for [PointsCompositionRules] tournaments.
 */
@Composable
private fun PointsBasedShipTypeSelector(
    composition: Composition,
    rules: PointsCompositionRules,
    selectionModel: SingleItemSelectionModel,
    currentShipType: ShipType?,
    onShipTypeSelected: (ShipType) -> Unit,
    onEditingCancelled: () -> Unit
) {
    val usedShipTypesWithoutCurrent = remember(composition, currentShipType) {
        composition.activeOrNullShipTypes() - currentShipType
    }
    val marginalShipCostComputation = remember(rules, usedShipTypesWithoutCurrent) {
        rules.marginalShipCostComputation(usedShipTypesWithoutCurrent)
    }

    val autoSuggest = remember(rules, marginalShipCostComputation) {
        AutoSuggest { text ->
            TheorycrafterContext.autoSuggest.shipTypes(text)?.filter { rules.isShipLegal(it) }
        }.onEmptyQueryReturn {
            val usedPointsWithoutCurrent = rules.shipsCost(usedShipTypesWithoutCurrent).sum()
            val remainingPoints = rules.maxCompositionCost - usedPointsWithoutCurrent
            val marginalCostByShipType = rules.legalShips.associateWith {
                marginalShipCostComputation.marginalCostOf(it)
            }
            rules.legalShips
                .filter { marginalCostByShipType[it]!! <= remainingPoints }
                .sortedByDescending { marginalCostByShipType[it] }
        }
    }

    Selector(
        onItemSelected = onShipTypeSelected,
        onEditingCancelled = onEditingCancelled,
        autoSuggest = autoSuggest,
        suggestedItemContent = { shipType, _ ->
            DefaultSuggestedEveItemTypeIcon(shipType)
            Text(text = shipType.shortName())
            Spacer(Modifier.weight(1f).widthIn(min = TheorycrafterTheme.spacing.medium))
            Text(text = marginalShipCostComputation.marginalCostOf(shipType).toString())
        },
        autoSuggestHorizontalAnchorPadding = DpRect(
            left = AutoSuggestHorizontalAnchorPaddingWithIcon.left,
            right = TheorycrafterTheme.spacing.horizontalEdgeMargin -
                    COST_COLUMN_PADDING.calculateEndPadding(LayoutDirection.Ltr),
            top = 0.dp,
            bottom = 0.dp
        ),
        hint = "Ship type",
        selectNextPrimaryRow = {
            selectionModel.selectNext()
        }
    )
}


/**
 * A ship type selection widget for [DraftCompositionRules] tournaments.
 */
@Composable
private fun PositionBasedShipTypeSelector(
    shipIndex: Int,
    rules: DraftCompositionRules,
    selectionModel: SingleItemSelectionModel,
    onShipTypeSelected: (ShipType) -> Unit,
    onEditingCancelled: () -> Unit
) {
    val position = rules.positions.getOrNull(shipIndex)
    val autoSuggest = remember(position, shipIndex) {
        val shipTypesForPosition = position?.legalShipTypes ?: rules.allLegalShipTypes
        val sortedShipTypesForPosition = shipTypesForPosition.sortedBy { it.name }
        AutoSuggest { text ->
            TheorycrafterContext.autoSuggest
                .shipTypes(text)
                ?.filter { it in shipTypesForPosition }
        }.onEmptyQueryReturn {
            sortedShipTypesForPosition
        }
    }

    Selector(
        onItemSelected = onShipTypeSelected,
        onEditingCancelled = onEditingCancelled,
        autoSuggest = autoSuggest,
        suggestedItemContent = { shipType, _ ->
            DefaultSuggestedEveItemTypeIcon(shipType)
            Text(text = shipType.shortName())
        },
        autoSuggestHorizontalAnchorPadding = DpRect(
            left = AutoSuggestHorizontalAnchorPaddingWithIcon.left,
            right = -TheorycrafterTheme.spacing.horizontalEdgeMargin,
            top = 0.dp,
            bottom = 0.dp
        ),
        hint = if (position != null) "${position.prefix} ${position.name}" else "Ship type",
        selectNextPrimaryRow = {
            selectionModel.selectNext()
        }
    )
}


/**
 * A fit selector widget.
 */
@Composable
private fun FitSelector(
    shipType: ShipType,
    selectionModel: SingleItemSelectionModel,
    onFitSelected: (FitHandle?) -> Unit,
    onEditingCancelled: () -> Unit
) {
    val autoSuggest = remember(TheorycrafterContext.fits.handlesKey) {
        AutoSuggest { text ->
            val fits = TheorycrafterContext.queryFits(text) ?: TheorycrafterContext.fits.handles
            buildList {
                add(null)
                for (fit in fits) {
                    if (fit.shipTypeId == shipType.itemId)
                        add(fit)
                }
            }
        }
    }

    Selector(
        onItemSelected = onFitSelected,
        onEditingCancelled = onEditingCancelled,
        autoSuggest = autoSuggest,
        suggestedItemContent = { fitHandle, _ ->
            Text(text = fitHandle?.name ?: "None")
        },
        autoSuggestHorizontalAnchorPadding = AutoSuggestHorizontalAnchorPadding,
        hint = "Fit name",
        selectNextPrimaryRow = {
            selectionModel.selectNext()
        }
    )
}



/**
 * The row displaying the "Empty Ship Slot".
 */
@Composable
private fun GridScope.EmptyShipSlotRow(
    composition: Composition,
    shipIndex: Int,
    selectionModel: SingleItemSelectionModel,
    actions: ShipSlotActions
) {
    val freeSlotTournament = composition.tournament.isFreeSlotTournament()
    val contextActions = if (!freeSlotTournament || (shipIndex == composition.visibleSlotCount - 1))
        emptyList()
    else
        remember(actions) {
            listOf(
                SlotContextAction.insertEmptySlot(actions),
                SlotContextAction.removeEmptySlot(actions),
            )
        }

    SlotRow(
        contextActions = contextActions,
        editedRowContent = { onEditingCompleted ->
            EditedShipTypeRowContent(
                composition = composition,
                shipIndex = shipIndex,
                selectionModel = selectionModel,
                currentShipType = null,
                onShipSelected = {
                    actions.setShip(it)
                    onEditingCompleted()
                },
                onEditingCancelled = onEditingCompleted
            )
        }
    ) {
        EmptyShipRowContent(
            composition = composition,
            shipIndex = shipIndex
        )
    }
}


/**
 * The content of a slot with no ship.
 */
@Composable
private fun GridScope.GridRowScope.EmptyShipRowContent(
    composition: Composition,
    shipIndex: Int,
) {
    emptyCell(GridCols.STATE_ICON)
    emptyCell(GridCols.SHIP_ICON)

    val rules = composition.tournament.rules.compositionRules
    val text = when {
        rules is DraftCompositionRules ->
            if (shipIndex in rules.positions.indices) {
                val pos = rules.positions[shipIndex]
                "${pos.prefix} ${pos.name}"
            }
            else
                "Replacement Ship"
        else -> "Empty Ship Slot"
    }
    cell(GridCols.SHIP, colSpan = 3) {
        Text(
            text = text,
            style = LocalTextStyle.current.copy(fontWeight = FontWeight.ExtraLight),
        )
    }
}


/**
 * Returns the active ships and their corresponding fits.
 */
@Composable
fun activeFitsByShip(composition: Composition): Map<Composition.Ship, Fit> {
    val fits = TheorycrafterContext.fits
    val shipsWithFitHandles by remember(composition) {
        derivedStateOf {
            composition.ships.mapNotNull {
                val fitId = it?.fitId ?: return@mapNotNull null
                val fitHandle = fits.handleById(fitId) ?: return@mapNotNull null
                Pair(it, fitHandle)
            }
        }
    }

    val fitByShip = produceState(initialValue = emptyMap(), shipsWithFitHandles, fits.handlesKey) {
        value = withContext(Dispatchers.Default) {
            buildMap {
                for ((ship, fitHandle) in shipsWithFitHandles) {
                    if (!fitHandle.isDeleted)
                        put(ship, fits.engineFitOf(fitHandle))
                }
            }
        }
    }.value

    // We must filter the inactive ships last, so that when the active state of a ship changes,
    // the caller receives the updated list on the very first recomposition.
    // If instead we filter in shipsWithFits, the first recomposition of the caller after a change in active state
    // (assuming it reads the active state) would happen with the previous list because produceState hasn't finished
    // producing the new list yet.
    return remember(fitByShip) {
        derivedStateOf {
            fitByShip.filterKeys { it.active }
        }
    }.value
}


/**
 * A row with the composition summary (number of ships, cost etc.)
 */
@Composable
private fun SummationRow(
    composition: Composition,
    shipsCost: List<Int>,
    modifier: Modifier
) {
    SimpleGridHeaderRow(
        modifier = modifier,
        columnWidths = ColumnWidths,
        defaultCellContentAlignment = ColumnAlignment::get
    ) {
        val rules = composition.tournament.rules.compositionRules
        val activeFitsByShip = activeFitsByShip(composition)
        val activeFits = activeFitsByShip.values

        EmptyCell(GridCols.STATE_ICON)
        EmptyCell(GridCols.SHIP_ICON)

        if (rules is PointsCompositionRules) {
            val shipCount = composition.ships.count { it?.active == true }
            val errorStyle = TextStyle(color = TheorycrafterTheme.colors.base().errorContent)
            TextCell(
                index = GridCols.SHIP,
                text = "$shipCount ships",
                style = if (shipCount > rules.maxCompositionSize) errorStyle else LocalTextStyle.current
            )

            val totalShipCost = shipsCost.sum()
            TextCell(
                index = GridCols.COST,
                text = "$totalShipCost pts",
                style = if (totalShipCost > rules.maxCompositionCost) errorStyle else LocalTextStyle.current,
                modifier = defaultCellModifier.then(COST_COLUMN_PADDING_MODIFIER)
            )
        } else {
            EmptyCell(GridCols.SHIP)
            EmptyCell(GridCols.COST)
        }

        EmptyCell(GridCols.FIT)

        TextCell(
            index = GridCols.EHP,
            text = activeFits.sumOf { it.defenses.ehp }.asHitPoints(ehp = true),
        )

        EmptyCell(GridCols.ACTIVE_TANK)

        TextCell(
            index = GridCols.DPS,
            text = activeFits.sumOf { it.firepower.totalDps }.asDps(),
        )
    }
}


/**
 * Displays and allows editing the composition note.
 */
@Composable
private fun CompositionNotes(
    composition: Composition,
    modifier: Modifier
) {
    TheorycrafterTheme.TextField(
        modifier = modifier,
        value = composition.note,
        minLines = 2,
        onValueChange = { composition.note = it },
        placeholder = { Text("Notes") }
    )
}


/**
 * The selection model for the given composition.
 */
private class CompositionShipsSelectionModel(
    val composition: Composition,
): SingleItemSelectionModel(initialSelectedIndex = 0) {

    override val maxSelectableIndex: Int
        get() = composition.visibleSlotCount - 1

}