package theorycrafter.ui.tournaments

import androidx.compose.foundation.layout.*
import androidx.compose.foundation.lazy.LazyListState
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.compose.material.DropdownMenuState
import androidx.compose.material.Text
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.input.key.Key
import androidx.compose.ui.platform.LocalClipboardManager
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.text.style.TextAlign
import compose.input.*
import compose.utils.*
import compose.widgets.*
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.async
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import theorycrafter.TestTags
import theorycrafter.TheorycrafterContext
import theorycrafter.formats.loadCompositionFromXml
import theorycrafter.storage.createDuplicateFitDetector
import theorycrafter.tournaments.Composition
import theorycrafter.tournaments.StoredComposition
import theorycrafter.tournaments.Tournament
import theorycrafter.ui.*
import theorycrafter.ui.widgets.*
import theorycrafter.utils.*
import java.io.File


/**
 * The panel displaying the list of compositions.
 */
@Composable
fun CompositionsPanel(
    tournament: Tournament,
    onShowComposition: (Composition) -> Unit,
    focusCompositionEditor: () -> Unit,
) {
    var newlyCreatedComposition: Composition? by remember { mutableStateOf(null) }

    var searchText by remember { mutableStateOf("") }.also {
        if (newlyCreatedComposition != null)
            it.value = ""
    }
    val queryIsEmpty = searchText.isEmpty()

    val filteredComps by remember(tournament.compositions) {
        derivedStateOf {
            val query = searchText
            if (query.isEmpty())
                tournament.compositions
            else
                tournament.compositions.filter { it.name.contains(query, ignoreCase = true) }
        }
    }

    val lazyListState = rememberLazyListState()
    val listState = remember(filteredComps, lazyListState, queryIsEmpty) {
        CompositionsListState(filteredComps, lazyListState, queryIsEmpty)
    }

    val selectionModel = listState.selectionModel
    val coroutineScope = rememberCoroutineScope()
    val dialogs = LocalStandardDialogs.current

    // Select the newly created composition and start renaming it
    LaunchedEffect(listState, newlyCreatedComposition) {
        if (newlyCreatedComposition != null) {
            selectionModel.selectItem(newlyCreatedComposition)
            listState.compositionBeingRenamed = newlyCreatedComposition
        }
    }

    // When the filtered list changes, select the first item
    LaunchedEffect(filteredComps) {
        if (filteredComps.isEmpty())
            selectionModel.clearSelection()
        else if (newlyCreatedComposition == null)
            selectionModel.selectIndex(0)
    }

    // When a selected fit is removed from the end, select the last one instead
    LaunchedEffect(listState.compositions.size) {
        val selectedIndex = selectionModel.selectedIndex
        if ((selectedIndex != null) && (selectedIndex > listState.compositions.lastIndex))
            selectionModel.selectIndex(listState.compositions.lastIndex)
    }

    fun onCompositionCreated(composition: Composition) {
        newlyCreatedComposition = composition
        onShowComposition(composition)
    }

    fun onCreateNewComposition() {
        val newComp = tournament.addEmptyComposition("New Composition")
        onCompositionCreated(newComp)
    }
    LocalKeyShortcutsManager.current.register(
        shortcut = CreateNewCompositionKeyShorcut,
        action = ::onCreateNewComposition
    )

    val clipboardManager = LocalClipboardManager.current
    val snackbarHost = LocalSnackbarHost.current
    fun onPasteCompositionFromClipboard() {
        coroutineScope.launch {
            val text = clipboardManager.getText()?.text
            if (text == null) {
                snackbarHost.showSnackbar("No text in the clipboard")
                return@launch
            }
            val composition = kotlin.runCatching {
                with(TheorycrafterContext.eveData) {
                    compositionFromClipboardText(text, tournament)
                }
            }.getOrNull()
            if (composition == null) {
                snackbarHost.showSnackbar("No composition detected in the clipboard")
                return@launch
            }

            val newComp = tournament.addComposition(composition)
            onCompositionCreated(newComp)
        }
    }
    LocalKeyShortcutsManager.current.register(
        shortcut = PasteCompositionFromClipboardKeyShortcut,
        action = ::onPasteCompositionFromClipboard
    )

    var showImportFileSelectionDialog by remember { mutableStateOf(false) }
    fun importCompositionFrom(file: File) {
        coroutineScope.launch {
            val duplicateFitDetectorDeferred = async { createDuplicateFitDetector(TheorycrafterContext.fits) }
            val result = withContext(Dispatchers.IO) {
                file.reader().use {
                    loadCompositionFromXml(eveData = TheorycrafterContext.eveData, reader = it)
                }
            }
            if (result.ships.isEmpty()) {
                dialogs.showErrorDialog("The specified file does not contain a valid composition")
                return@launch
            }
            val duplicateFitDetector = duplicateFitDetectorDeferred.await()
            var existingFits = 0
            var newFits = 0
            val fitIdByStoredFit = result.ships
                .mapNotNull { it.fit }
                .associateWith { storedFit ->
                    val existingFit = duplicateFitDetector.findDuplicateOf(storedFit)
                    if (existingFit != null) {
                        existingFits += 1
                        existingFit.id
                    } else {
                        // Create new fit
                        newFits += 1
                        TheorycrafterContext.fits.add(storedFit).fitId
                    }
                }
            val ships = buildList {
                for (ship in result.ships) {
                    val fitId = fitIdByStoredFit[ship.fit]
                    add(
                        StoredComposition.StoredCompositionShip(
                            shipTypeId = ship.shipTypeId,
                            fitId = fitId,
                            active = ship.active
                        )
                    )
                }
            }
            val storedComposition = StoredComposition(
                name = result.name ?: file.name.substringBeforeLast("."),
                note = result.note ?: "",
                ships = ships
            )
            val newComp = tournament.addComposition(storedComposition)

            launch {
                snackbarHost.showSnackbar("Imported $newFits new fits and used $existingFits existing ones")
            }
            onCompositionCreated(newComp)
        }
    }

    fun onEditCompositionName(composition: Composition) {
        listState.compositionBeingRenamed = composition
    }

    fun onDuplicateComposition(composition: Composition) {
        coroutineScope.launch {
            val copyName = copyName(composition.name, allNames = tournament.compositions.asSequence().map { it.name })
            val newComp = tournament.duplicateComposition(composition, name = copyName)
            onCompositionCreated(newComp)
        }
    }

    fun onDeleteComposition(composition: Composition) {
        dialogs.showConfirmDialog(
            text = "Delete ${composition.name}?",
            confirmText = "Delete",
            onConfirm = {
                coroutineScope.launch {
                    tournament.deleteCompositions(listOf(composition))
                }
            }
        )
    }

    val compositionExporter = rememberCompositionExporter()

    Column(
        modifier = Modifier
            .fillMaxSize()
            .moveSelectionWithKeys(
                selectionModel = selectionModel,
                itemsInPage = lazyListState::itemsInPage,
                onPreview = true
            )
            .onKeyShortcut(Key.F2, onPreview = true) {
                selectionModel.selectedItem()?.let {
                    onEditCompositionName(it)
                }
            }
            .onKeyShortcut(KeyShortcut.DeleteItem) {
                selectionModel.selectedItem()?.let {
                    onDeleteComposition(it)
                }
            }
            .onKeyShortcut(KeyShortcut.anyEnter(), onPreview = true) {
                // Ignore and don't consume ENTER when editing a composition's name.
                if (listState.compositionBeingRenamed != null) {
                    consumeEvent = false
                    return@onKeyShortcut
                }
                selectionModel.selectedItem()?.let {
                    onShowComposition(it)
                }
            }
            .onKeyShortcut(KeyShortcut.CopyToClipboard, onPreview = true) {
                listState.selectionModel.selectedItem()?.let {
                    compositionExporter.copyToClipboard(it)
                }
            }
    ) {
        CompositionsSearchField(
            modifier = Modifier
                .fillMaxWidth()
                .padding(bottom = TheorycrafterTheme.spacing.small)
                .padding(horizontal = TheorycrafterTheme.spacing.horizontalEdgeMargin),
            searchText = searchText,
            onSearched = {
                searchText = it
            }
        )

        val compositionsListFocusRequester = remember { FocusRequester() }
        CompositionsList(
            state = listState,
            modifier = Modifier
                .fillMaxWidth()
                .weight(1f)
                .focusRequester(compositionsListFocusRequester),
            onShowComposition = onShowComposition,
            onRenameComposition = { comp, name ->
                comp.name = name
            },
            onEditingNameFinished = { comp ->
                listState.compositionBeingRenamed = null
                if (comp == newlyCreatedComposition) {
                    newlyCreatedComposition = null
                    focusCompositionEditor()
                }
                else {
                    compositionsListFocusRequester.requestFocus()
                }
            },
            onCopyComposition = compositionExporter::copyToClipboard,
            onCopyCompositionWithOptions = compositionExporter::copyToClipboardWithOptions,
            onExportFitsToXml = compositionExporter::exportToXml,
            onEditCompositionName = ::onEditCompositionName,
            onDuplicateComposition = ::onDuplicateComposition,
            onDeleteComposition = ::onDeleteComposition,
        )

        Row(
            modifier = Modifier
                .fillMaxWidth()
                .padding(
                    horizontal = TheorycrafterTheme.spacing.horizontalEdgeMargin,
                    vertical = TheorycrafterTheme.spacing.medium
                ),
            horizontalArrangement = Arrangement.spacedBy(TheorycrafterTheme.spacing.small, Alignment.End)
        ) {
            val tooltipPlacement = EasyTooltipPlacement.ElementTopCenter(
                offset = DpOffsetY(y = -TheorycrafterTheme.spacing.xxsmall)
            )
            TheorycrafterTheme.RaisedButtonWithText(
                text = "New Composition",
                onClick = ::onCreateNewComposition,
                modifier = Modifier
                    .testTag(TestTags.CompsList.NewCompositionButton)
                    .nonFocusable()
                    .tooltip(
                        text = "Create new composition",
                        keyShortcut = CreateNewCompositionKeyShorcut,
                        placement = tooltipPlacement
                    )
            )

            MenuButton(
                modifier = Modifier.nonFocusable(),
                offset = DpOffsetY(y = TheorycrafterTheme.spacing.xxxsmall),
                content = { onClick ->
                    TheorycrafterTheme.RaisedButtonWithText(
                        text = "Import Comp.",
                        onClick = onClick
                    )
                },
                menuContent = { onCloseMenu ->

                    @Composable
                    fun menuItem(
                        text: String,
                        icon: (@Composable () -> Unit)? = null,
                        keyShortcut: KeyShortcut? = null,
                        action: () -> Unit) {
                        MenuItem(
                            text = text,
                            icon = icon,
                            displayedKeyShortcut = keyShortcut,
                            reserveSpaceForKeyShortcut = true,
                            keyShortcutDescriptorWidth = TheorycrafterTheme.sizes.menuItemKeyShortcutWidthMedium,
                            onCloseMenu = onCloseMenu,
                            action = action
                        )
                    }

                    menuItem(
                        text = "Import from XML File…",
                        icon = { Icons.ImportFromFile() },
                        action = { showImportFileSelectionDialog = true }
                    )
                    menuItem(
                        text = "Paste from Clipboard",
                        icon = { Icons.Paste() },
                        keyShortcut = PasteCompositionFromClipboardKeyShortcut,
                        action = ::onPasteCompositionFromClipboard
                    )
                }
            )
        }
    }

    if (showImportFileSelectionDialog) {
        NativeReadFilePickerDialog(
            title = "Select XML composition file",
            filenameFilter = extensionFilenameFilter(ignoreCase = true, "xml"),
            onCompletedSelecting = { file ->
                if (file != null) {
                    importCompositionFrom(file)
                }
                showImportFileSelectionDialog = false
            }
        )
    }
}


/**
 * The compositions search field.
 */
@Composable
private fun CompositionsSearchField(
    modifier: Modifier,
    searchText: String,
    onSearched: (String) -> Unit
) {
    SearchField(
        modifier = modifier
            .requestInitialFocus(),
        searchText = searchText,
        placeholderText = "Composition name",
        focusKeyShortcut = KeyShortcut(Key.F, KeyboardModifierMatcher.Command),
        onSearched = onSearched
    )
}


/**
 * The state object for the [CompositionsList].
 */
private class CompositionsListState(


    /**
     * The compositions to display.
     */
    val compositions: List<Composition>,


    /**
     * The state of the actual lazy list displaying the compositions.
     */
    val lazyListState: LazyListState,


    /**
     * Whether the composition query is empty.
     */
    val isQueryEmpty: Boolean


) {


    /**
     * The selection model.
     */
    val selectionModel = SingleItemListSelectionModel(compositions)


    /**
     * The composition currently being renamed; `null` if none.
     */
    var compositionBeingRenamed: Composition? by mutableStateOf(null)


}


/**
 * The list of compositions itself.
 */
@Composable
private fun CompositionsList(
    state: CompositionsListState,
    modifier: Modifier = Modifier,
    onShowComposition: (Composition) -> Unit,
    onRenameComposition: (Composition, String) -> Unit,
    onEditingNameFinished: (Composition) -> Unit,
    onCopyComposition: (Composition) -> Unit,
    onCopyCompositionWithOptions: (Composition) -> Unit,
    onExportFitsToXml: (Composition) -> Unit,
    onEditCompositionName: (Composition) -> Unit,
    onDuplicateComposition: (Composition) -> Unit,
    onDeleteComposition: (Composition) -> Unit,
) {
    val compositions = state.compositions
    val selectionModel = state.selectionModel

    if (compositions.isEmpty()) {
        Text(
            text = if (state.isQueryEmpty)
                "No compositions yet"
            else
                "No matching compositions",
            textAlign = TextAlign.Center,
            modifier = modifier
                .fillMaxWidth()
                .padding(top = TheorycrafterTheme.spacing.xxlarge),
        )

        return
    }

    // Need to Box up so that ContextMenu has the same bounds as the list
    Box(modifier) {
        val contextMenuState by remember { mutableStateOf(DropdownMenuState()) }

        ScrollShadow(state.lazyListState, top = true, bottom = true)
        LazyColumnExt(
            state = state.lazyListState,
            selection = selectionModel,
            selectionBackground = TheorycrafterTheme.colors.selectedItemBackground(),
            modifier = Modifier
                .onOpenContextMenu { position ->
                    state.lazyListState.itemIndexAt(position.y)?.let { clickedIndex ->
                        selectionModel.selectIndex(clickedIndex)
                        contextMenuState.status = DropdownMenuState.Status.Open(position)
                    }
                }
                .focusWhenClicked()
        ) {
            items(count = compositions.size) { index ->
                val composition = compositions[index]
                if (composition == state.compositionBeingRenamed) {
                    CompNameEditor(
                        name = composition.name,
                        modifier = Modifier.fillMaxWidth(),
                        onRename = { name ->
                            onRenameComposition(composition, name)
                        },
                        onEditingFinished = {
                            onEditingNameFinished(composition)
                        }
                    )
                }
                else {
                    CompositionListItem(
                        name = composition.name,
                        modifier = Modifier
                            .fillMaxWidth()
                            .onMousePress(clickCount = ClickCount.DOUBLE, consumeEvent = true) {
                                onShowComposition(composition)
                            }
                            .highlightOnHover()
                    )
                }
            }
        }

        ContextMenu(contextMenuState) {

            @Composable
            fun menuItem(
                text: String,
                icon: (@Composable () -> Unit)? = EmptyIcon,
                keyShortcut: KeyShortcut? = null,
                action: (Composition) -> Unit
            ) {
                MenuItem(
                    text = text,
                    icon = icon,
                    displayedKeyShortcut = keyShortcut,
                    reserveSpaceForKeyShortcut = true,
                    onCloseMenu = contextMenuState::close,
                    action = {
                        selectionModel.selectedItem()?.let {
                            action.invoke(it)
                        }
                    }
                )
            }

            menuItem("Copy", icon = { Icons.Copy() }, KeyShortcut.CopyToClipboard) {
                onCopyComposition(it)
            }
            menuItem("Copy with Options…", icon = { Icons.CopyWithOptions() }, keyShortcut = null) {
                onCopyCompositionWithOptions(it)
            }
            menuItem("Export to XML File…", icon = { Icons.Share() }, keyShortcut = null) {
                onExportFitsToXml(it)
            }

            MenuSeparator()
            menuItem("Rename", icon = { Icons.Edit() }, KeyShortcut.RenameItem) {
                onEditCompositionName(it)
            }
            menuItem("Duplicate", icon = { Icons.Duplicate() }, keyShortcut = null) {
                onDuplicateComposition(it)
            }
            menuItem("Delete", icon = { Icons.Delete() }, KeyShortcut.DeleteItem.first()) {
                onDeleteComposition(it)
            }
        }
    }
}


/**
 * An item in the compositions list.
 */
@Composable
private fun CompositionListItem(
    name: String,
    modifier: Modifier = Modifier
) {
    VerticallyCenteredRow(
        modifier = modifier
            .padding(
                horizontal = CompNameItemPaddingHorizontal,
                vertical = CompNameItemPaddingVertical
            ),
    ) {
        SingleLineText(name)
    }
}


/**
 * The horizontal padding of the composition name item.
 */
private val CompNameItemPaddingHorizontal = TheorycrafterTheme.spacing.horizontalEdgeMargin


/**
 * The vertical padding of the composition name item.
 */
private val CompNameItemPaddingVertical = TheorycrafterTheme.spacing.xsmall


/**
 * The composition name being edited.
 */
@Composable
private fun CompNameEditor(
    name: String,
    modifier: Modifier,
    onRename: (String) -> Unit,
    onEditingFinished: () -> Unit,
) {
    val padding = PaddingValues(
        horizontal = CompNameItemPaddingHorizontal,
        vertical = CompNameItemPaddingVertical
    )
    ItemNameEditor(
        itemName = name,
        modifier = modifier
            .padding(padding)
            .bringIntoViewWhenFocusedWithMargins(padding),
        onRename = onRename,
        onEditingFinished = onEditingFinished
    )
}


/**
 * The key shortcut to create a new composition.
 */
private val CreateNewCompositionKeyShorcut = MainWindowKeyShortcuts.CreateNewFit


/**
 * The key shortcut to paste a composition from the clipboard.
 */
private val PasteCompositionFromClipboardKeyShortcut = MainWindowKeyShortcuts.PasteFitFromClipboard
