package theorycrafter.ui.tournaments

import androidx.compose.foundation.layout.*
import androidx.compose.material.SnackbarHostState
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.input.key.Key
import androidx.compose.ui.window.WindowPosition
import compose.input.KeyShortcut
import compose.input.KeyboardModifierMatcher
import eve.data.EveData
import eve.data.typeid.isCommandBurst
import kotlinx.coroutines.launch
import org.jetbrains.skiko.OS
import org.jetbrains.skiko.hostOs
import theorycrafter.*
import theorycrafter.fitting.Fit
import theorycrafter.tournaments.Composition
import theorycrafter.tournaments.Tournament
import theorycrafter.tournaments.TournamentDescriptor
import theorycrafter.tournaments.isLogistics
import theorycrafter.ui.*
import theorycrafter.ui.fiteditor.KeyShortcutsIcon
import theorycrafter.ui.widgets.*


/**
 * The tournament window.
 */
@Composable
fun TournamentWindow() {
    val windowManager = LocalTheorycrafterWindowManager.current
    val state = windowManager.tournamentWindowState
    if (state !is TournamentWindowState.Open)
        return

    val tournamentDesc = state.tournamentDescriptor
    val windowState = rememberWindowStateAndUpdateSettings(
        windowSettings = state.windowSettings,
        defaultPosition = { WindowPosition.PlatformDefault },
        defaultSize = { TheorycrafterTheme.sizes.tournamentWindowDefaultSize },
    )

    TheorycrafterWindow(
        title = tournamentDesc.name,
        onCloseRequest = windowManager::closeTournamentWindow,
        state = windowState
    ) {
        key(tournamentDesc.id) {
            TournamentWindowContent(tournamentDesc)
        }

        LaunchedEffect(state, window) {
            state.window = window
        }

        LaunchedEffect(tournamentDesc) {
            window.toFront()  // The tournament may have changed
        }
    }
}


/**
 * The state of a tournament window.
 */
sealed interface TournamentWindowState {


    /**
     * The window is closed.
     */
    @Stable
    data object Closed: TournamentWindowState


    /**
     * The window is open, showing the given tournament.
     */
    @Stable
    class Open(


        /**
         * The descriptor of the tournament being displayed.
         */
        val tournamentDescriptor: TournamentDescriptor,


        /**
         * The window settings.
         */
        val windowSettings: TheorycrafterSettings.WindowSettings


    ): TheorycrafterWindowInfo(), TournamentWindowState


}


/**
 * The content of the tournament window.
 */
@Composable
fun TournamentWindowContent(tournamentDescriptor: TournamentDescriptor) {
    val tournament by produceState<Tournament?>(initialValue = null, key1 = tournamentDescriptor) {
        value = TheorycrafterContext.tournaments.loadTournament(tournamentDescriptor)
    }
    val snackbarHost = remember { SnackbarHostState() }
    CompositionLocalProvider(LocalSnackbarHost provides snackbarHost) {
        val (comp, setComp) = remember { mutableStateOf<Composition?>(null) }
        LaunchedEffect(comp?.deleted) {
            if ((comp != null) && comp.deleted)
                setComp(null)
        }

        val compositionEditorFocusRequester = remember { FocusRequester() }

        ThreePanelScaffold(
            modifier = Modifier.fillMaxSize(),
            left = {
                TitledPanel(
                    title = "Compositions",
                    modifier = Modifier.width(TheorycrafterTheme.sizes.compositionsPanelWidth)
                ) {
                    tournament?.let {
                        CompositionsPanel(
                            tournament = it,
                            onShowComposition = setComp,
                            focusCompositionEditor = {
                                compositionEditorFocusRequester.requestFocus()
                            }
                        )
                    }
                }
            },
            middle = {
                if (comp == null)
                    return@ThreePanelScaffold
                key(comp) {
                    TitledPanel(
                        title = comp.name,
                        actionsButton = { CompositionActionButton(comp) }
                    ) {
                        Column(Modifier.fillMaxSize()) {
                            CompositionEditor(
                                composition = comp,
                                modifier = Modifier
                                    .fillMaxWidth()
                                    .weight(1f)
                                    .focusRequester(compositionEditorFocusRequester)
                            )

                            KeyShortcutsIcon(
                                modifier = Modifier
                                    .align(Alignment.End)
                                    .padding(
                                        end = TheorycrafterTheme.spacing.horizontalEdgeMargin,
                                        bottom = TheorycrafterTheme.spacing.horizontalEdgeMargin,
                                        top = TheorycrafterTheme.spacing.small
                                    )
                            ) {
                                KeyShortcutsTooltip(COMPOSITION_KEY_SHORTCUTS_TOOLTIP_CONTENTS)
                            }
                        }
                    }
                }
            },
            snackbarHostState = snackbarHost
        )
    }
}


/**
 * The action button for the composition.
 */
@Composable
private fun CompositionActionButton(composition: Composition) {
    val snackbarState = LocalSnackbarHost.current
    val coroutineScope = rememberCoroutineScope()
    val compositionExporter = rememberCompositionExporter()

    ActionsMenuButton(contentDescription = "Composition Actions") {
        menuItem(
            text = "Copy Composition to Clipboard",
            icon = { Icons.Copy() },
            keyShortcut = CopyCompositionToClipboardKeyShortcut
        ) {
            compositionExporter.copyToClipboard(composition)
        }

        menuItem(
            text = "Copy to Clipboard with Options…",
            icon = { Icons.CopyWithOptions() },
            keyShortcut = null
        ) {
            compositionExporter.copyToClipboardWithOptions(composition)
        }

        menuItem(
            text = "Export Composition to XML File…",
            icon = { Icons.Share() },
            keyShortcut = null
        ) {
            compositionExporter.exportToXml(composition)
        }

        separator()

        menuItem(
            text = "Apply Command and Repair Effects",
            enabled = composition.ships.any { it?.fitId != null },
            icon = { Icons.ApplyCrossFitFriendlyEffects() },
            keyShortcut = ApplyCrossFitFriendlyEffectsKeyShortcut
        ) {
            coroutineScope.launch {
                with(TheorycrafterContext.eveData) {
                    composition.applyCrossShipAssistiveEffects(snackbarState)
                }
            }
        }
    }
}


/**
 * Applies the relevant effects from every ship in the composition to all other ships.
 */
context(EveData)
private suspend fun Composition.applyCrossShipAssistiveEffects(snackbarState: SnackbarHostState) {
    data class ShipWithFit(val ship: Composition.Ship, val fit: Fit)
    val fitContext = TheorycrafterContext.fits
    val shipsWithFits = ships
        .mapNotNull { ship ->
            if ((ship == null) || !ship.active) return@mapNotNull null
            val fitId = ship.fitId ?: return@mapNotNull null
            val fitHandle = fitContext.handleById(fitId) ?: return@mapNotNull null
            ShipWithFit(ship, fitContext.engineFitOf(fitHandle))
        }
    val shipsWithUniqueFits = shipsWithFits.distinctBy { it.fit }

    val neededCommandEffects = mutableSetOf<Pair<ShipWithFit, ShipWithFit>>()
    val neededRepairEffects = mutableListOf<Pair<ShipWithFit, ShipWithFit>>()

    for (source in shipsWithFits) {
        val hasCommandBursts = source.fit.modules.all.any { it.type.isCommandBurst() }
        val isLogi = source.fit.ship.type.isLogistics

        for (target in shipsWithUniqueFits) {
            if (target.ship === source.ship)
                continue

            if (hasCommandBursts)
                neededCommandEffects.add(source to target)

            if (isLogi)
                neededRepairEffects.add(source to target)
        }
    }

    var removedEffectCount = 0
    var addedCommandEffectCount = 0
    var addedRepairEffectCount = 0
    fitContext.modifyAndSave {
        val neededRepairEffectCountBySourceAndTargetFit = neededRepairEffects.groupingBy { (source, target) ->
            Pair(source.fit, target.fit)
        }.eachCount()

        // Clear existing assistive effects
        for (target in shipsWithFits) {
            for (commandEffect in target.fit.commandEffects.toList()) {
                val isCommandEffectNeeded = neededCommandEffects.any { (s, t) ->
                    (commandEffect.source == s.fit) && (commandEffect.target == t.fit)
                }
                if (!isCommandEffectNeeded) {
                    target.fit.removeCommandEffect(commandEffect)
                    removedEffectCount++
                }
            }

            val currentCountBySource = target.fit.friendlyEffects.groupingBy { it.source }.eachCount().toMutableMap()
            for (friendlyEffect in target.fit.friendlyEffects.toList()) {
                if (friendlyEffect.isByAuxiliaryFit) {
                    for (module in friendlyEffect.affectingModules) {
                        target.fit.removeModuleEffect(module)
                        removedEffectCount++
                    }
                    continue
                }
                val neededCount = neededRepairEffectCountBySourceAndTargetFit[friendlyEffect.source to friendlyEffect.target] ?: 0
                val currentCount = currentCountBySource[friendlyEffect.source] ?: 0
                if (currentCount > neededCount) {
                    target.fit.removeFriendlyEffect(friendlyEffect)
                    currentCountBySource[friendlyEffect.source] = currentCount - 1
                    removedEffectCount++
                }
            }
        }

        // Add command effects
        for ((source, target) in neededCommandEffects) {
            if (target.fit.commandEffects.none { it.source == source.fit }) {
                target.fit.addCommandEffect(source.fit)
                addedCommandEffectCount++
            }
        }

        // Add friendly effects
        for ((source, target) in neededRepairEffects) {
            val neededCount = neededRepairEffectCountBySourceAndTargetFit[source.fit to target.fit] ?: 0
            val currentCount = target.fit.friendlyEffects.count { it.source == source.fit }
            if (neededCount > currentCount) {
                target.fit.addFriendlyEffect(source.fit)
                addedRepairEffectCount++
            }
        }
    }

    if (addedCommandEffectCount + addedRepairEffectCount + removedEffectCount == 0)
        snackbarState.showSnackbar("No new effects were applied or removed")
    else {
        snackbarState.showSnackbar(
            """
                $addedCommandEffectCount command effects were applied
                $addedRepairEffectCount repair effects were applied
                $removedEffectCount existing effects were removed
            """.trimIndent()
        )
    }
}


/**
 * The legend of the keyboard shortcuts in the composition editor.
 */
private val COMPOSITION_KEY_SHORTCUTS_TOOLTIP_CONTENTS = listOf(
    KeyShortcutDescription(
        description = "Edit ship",
        DisplayedKeyShortcut("enter", verticalShiftMultiplier = LowercaseTextShiftMultiplier),
        DisplayedKeyShortcut("F2")
    ),
    KeyShortcutDescription(
        description = "Edit fit",
        DisplayedKeyShortcut(
            keys = listOf(if (hostOs == OS.MacOS) "⌥" else "alt", "enter"),
            verticalShiftMultiplier = LowercaseTextShiftMultiplier),
    ),
    KeyShortcutDescription(
        description = "Clear slot",
        DisplayedKeyShortcut("del", verticalShiftMultiplier = 0.1f), // 'del' is taller than regular text
        DisplayedKeyShortcut("⌫", verticalShiftMultiplier = 0.05f)
    ),
    KeyShortcutDescription(
        description = "Toggle active state",
        DisplayedKeyShortcut("space", verticalShiftMultiplier = LowercaseTextShiftMultiplier),
        DisplayedKeyShortcut("T")
    ),
)


/**
 * The key shortcut to copy the currently displayed composition to the clipboard.
 */
val CopyCompositionToClipboardKeyShortcut = FitWindowKeyShortcuts.CopyFitToClipboard


/**
 * The key shortcut to apply cross-fit effects on all fits in the composition.
 */
val ApplyCrossFitFriendlyEffectsKeyShortcut = KeyShortcut(Key.E, KeyboardModifierMatcher.Command)
