package theorycrafter.ui.widgets

import androidx.compose.animation.core.animateFloatAsState
import androidx.compose.animation.core.tween
import androidx.compose.foundation.Canvas
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.LocalContentColor
import androidx.compose.material.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import eve.data.fractionAsPercentage
import theorycrafter.ui.TheorycrafterTheme
import theorycrafter.utils.darker


/**
 * A horizontal box displaying a percentage value.
 */
@Composable
fun PercentageView(


    /**
     * The value to display; a number between 0 and 1, inclusive.
     */
    value: Float,


    /**
     * The number of decimal digits to display.
     */
    decimals: Int,


    /**
     * Whether to animate value changes.
     */
    animate: Boolean = true,


    /**
     * The filled color.
     */
    filledColor: Color,


    /**
     * The unfilled color.
     */
    unfilledColor: Color = filledColor.darker(0.3f),


    /**
     * A modifier to apply to the top-level element.
     */
    modifier: Modifier = Modifier


) {
    if (value.isNaN() || value.isInfinite())
        throw IllegalArgumentException("Illegal value: $value")

    val displayedValue = if (animate)
        animateFloatAsState(
            targetValue = value,
            animationSpec = tween()
        ).value
    else
        value

    Box(
        modifier = modifier
            .border(
                width = Dp.Hairline,
                color = LocalContentColor.current.copy(alpha = 0.25f),
                shape = RoundedCornerShape(2.dp)
            )
            .clip(
                RoundedCornerShape(2.dp)
            )
    ) {
        // The slider
        Canvas(
            modifier = Modifier.matchParentSize()
        ) {
            drawRect(unfilledColor)
            drawRect(filledColor,
                topLeft = Offset.Zero,
                size = Size(
                    width = (size.width * displayedValue).coerceAtMost(size.width),
                    height = size.height
                )
            )
        }

        // The percentage text
        Text(
            text = TheorycrafterTheme.textStyles.withShadow(
                displayedValue.toDouble().let {
                    if (it > 9.99)
                        ">999%"
                    else
                        it.fractionAsPercentage(decimals)
                }
            ),
            maxLines = 1,
            overflow = TextOverflow.Visible,
            textAlign = TextAlign.Center,
            style = TheorycrafterTheme.textStyles.percentageView,
            modifier = Modifier
                .padding(4.dp)
                .fillMaxWidth()
                .align(Alignment.Center)
        )
    }
}
