/**
 * Utilities related to fonts.
 */

package theorycrafter.utils

import androidx.compose.material.LocalTextStyle
import androidx.compose.runtime.Composable
import androidx.compose.runtime.remember
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.platform.LocalFontFamilyResolver
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.font.FontFamily
import androidx.compose.ui.text.font.FontStyle
import androidx.compose.ui.text.font.FontSynthesis
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.platform.FontLoadResult
import androidx.compose.ui.text.platform.toSkFontStyle
import androidx.compose.ui.unit.Density
import org.jetbrains.skia.FontMetrics


/**
 * Returns a remembered font metrics for the current text style, density and font family resolver.
 */
@Composable
fun rememberLocalFontMetrics(textStyle: TextStyle = LocalTextStyle.current): FontMetrics {
    val fontFamilyResolver = LocalFontFamilyResolver.current
    val density = LocalDensity.current
    return remember(textStyle, density, fontFamilyResolver) {
        textStyle.fontMetrics(density, fontFamilyResolver)
    }
}


/**
 * Returns the font metrics for the [TextStyle].
 */
fun TextStyle.fontMetrics(density: Density, fontFamilyResolver: FontFamily.Resolver): FontMetrics {
    val res = org.jetbrains.skia.paragraph.TextStyle()

    fontStyle?.let {
        res.fontStyle = it.toSkFontStyle()
    }

    fontWeight?.let {
        res.fontStyle = res.fontStyle.withWeight(it.weight)
    }

    res.fontSize = with(density){ fontSize.toPx() }

    fontFamily?.let {
        val resolved = fontFamilyResolver.resolve(
            it,
            fontWeight ?: FontWeight.Normal,
            fontStyle ?: FontStyle.Normal,
            fontSynthesis ?: FontSynthesis.None
        ).value as FontLoadResult
        res.fontFamilies = resolved.aliases.toTypedArray()
    }

    return res.fontMetrics
}
