package theorycrafter.fitting


/**
 * Specifies the amount of adaptation cycles to set on a module.
 */
sealed interface AdaptationCycles {


    /**
     * Specifies that the maximum number of adaptation cycles should be set.
     */
    data object Maximum: AdaptationCycles


    /**
     * Specifies an exact number of adaptation cycles to be set.
     */
    class Number(val value: Int): AdaptationCycles


    /**
     * Returns the actual number of adaption cycles to be set.
     */
    val cycleCount: Int
        get() = when (this) {
            is Maximum -> Short.MAX_VALUE.toInt()
            is Number -> value
        }


    companion object {


        /**
         * Returns the [AdaptationCycles] of the given module.
         */
        fun of(module: Module): AdaptationCycles? {
            val count = module.adaptationCycles?.value ?: return null
            return if (count == Maximum.cycleCount) Maximum else Number(count)
        }


    }


}
