package theorycrafter.ui

import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.material.Divider
import androidx.compose.material.Text
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import compose.utils.VSpacer
import compose.widgets.SingleLineText
import theorycrafter.Theorycrafter
import theorycrafter.ui.widgets.DialogInitiallyFocusedButton
import theorycrafter.ui.widgets.InnerDialog
import java.awt.Desktop
import java.awt.Window
import java.text.SimpleDateFormat


/**
 * The "about" dialog.
 */
@Composable
fun AboutDialog(parentWindow: Window) {
    var showAboutDialog by remember { mutableStateOf(false) }
    fun dismiss() {
        showAboutDialog = false
    }

    if (showAboutDialog) {
        InnerDialog(
            confirmText = "Close",
            dismissText = null,
            onConfirm = ::dismiss,
            onDismiss = ::dismiss,
            initiallyFocusedButton= DialogInitiallyFocusedButton.Confirm,
            modifier = Modifier.width(TheorycrafterTheme.sizes.aboutDialogWidth)
        ) {
            AboutDialogContent(
                Modifier
                    .padding(horizontal = TheorycrafterTheme.spacing.medium)
                    .padding(top = TheorycrafterTheme.spacing.small)
                    .fillMaxWidth()
            )
        }

        LaunchedEffect(Unit) {
            parentWindow.toFront()
        }
    }

    LaunchedEffect(Unit) {
        with(Desktop.getDesktop()) {
            if (isSupported(Desktop.Action.APP_ABOUT))
                setAboutHandler { showAboutDialog = true }
        }
    }
}


/**
 * Quotes from Arilou'laleelay.
 */
private val ArilouQuotes = listOf(
    """
        Hello my little one. I am so pleased to see you!
        You have done well for yourself.
        It is gratifying.
    """.trimIndent(),
    """
        Hello my clever child. We have met again and I am pleased.
        Your people are so beautiful... so unspoiled.
    """.trimIndent(),
    """
        Did you desire this meeting, Captain, or did we?
    """.trimIndent(),
    """
        Hello. We are pleased to see you again.
        Have you any numbness yet?
        Good. It should be too soon for that. Too soon by far.
    """.trimIndent(),
    """
        Last night as you slept, I touched your face and you smiled!...
        but now you frown. A pity. Smiling is healthier.
    """.trimIndent(),
    """
        The enemy of your enemy is your friend... at least for a time.
    """.trimIndent(),
    """
        Human. You flail at the wind. Cease your violence.
        Would you hurt your pet? Your child? Your beast?
        Please... rethink your posture.
    """.trimIndent()
)


/**
 * The content of the "About" dialog.
 */
@Composable
fun AboutDialogContent(modifier: Modifier) {
    Column(
        modifier = modifier,
        horizontalAlignment = Alignment.CenterHorizontally
    ) {
        SingleLineText(
            text = "${Theorycrafter.AppName} ${Theorycrafter.Version}",
            style = TheorycrafterTheme.textStyles.largeHeading.copy(fontWeight = FontWeight.Bold),
        )

        VSpacer(TheorycrafterTheme.spacing.small)

        Text(
            text = SimpleDateFormat.getDateInstance().format(Theorycrafter.BuildTimestamp),
            textAlign = TextAlign.Center,
            style = TheorycrafterTheme.textStyles.smallHeading.copy(fontWeight = FontWeight.Normal),
        )

        VSpacer(TheorycrafterTheme.spacing.xlarge)

        Text(
            text = remember { ArilouQuotes.random() },
            textAlign = TextAlign.Center
        )

        VSpacer(TheorycrafterTheme.spacing.xlarge)

        Divider()

        VSpacer(TheorycrafterTheme.spacing.xlarge)

        Text(
            text = "${Theorycrafter.AppName} displays icons from EVE Online\n© 2014 CCP hf. All rights reserved.\n\n" +
                    "\"EVE\", \"EVE Online\", \"CCP\", and all related logos and images are trademarks or registered trademarks of CCP hf.",
            style = TheorycrafterTheme.textStyles.caption,
            textAlign = TextAlign.Center
        )

        VSpacer(TheorycrafterTheme.spacing.xlarge)

        Divider()

        VSpacer(TheorycrafterTheme.spacing.xlarge)

        Text(
            text = "${System.getProperty("java.runtime.name")} ${System.getProperty("java.version")}" +
                    " by ${System.getProperty("java.vendor")}",
            style = TheorycrafterTheme.textStyles.caption,
            textAlign = TextAlign.Center
        )
    }
}