package theorycrafter.tournaments.impl.ag

import eve.data.EveData
import eve.data.ModuleType
import eve.data.ShipType
import eve.data.typeid.isRemoteArmorRepairer
import theorycrafter.sharedapp.generated.resources.Res
import theorycrafter.sharedapp.generated.resources.ag7
import theorycrafter.tournaments.*
import theorycrafter.tournaments.PointsCompositionRules.MarginalShipCostComputation


/**
 * The descriptor for Anger Games 7 (2026).
 */
val AngerGames7 = TournamentDescriptor(
    id = "ag7",
    name = "Anger Games 7",
    shortName = "AG7",
    icon = Res.drawable.ag7,
    rulesFactory = ::angerGames7Rules,
)


/**
 * Maps ship type names to their point values.
 */
private val ShipCostByName = mapOf(
    "Impairor" to 1,
    "Ibis" to 2,
    "Velator" to 1,
    "Reaper" to 1,
    "Anathema" to 4,
    "Buzzard" to 4,
    "Helios" to 4,
    "Cheetah" to 4,
    "Hydra" to 4,
    "Bestower" to 4,
    "Sigil" to 4,
    "Badger" to 4,
    "Tayra" to 4,
    "Epithal" to 4,
    "Iteron Mark V" to 4,
    "Kryos" to 4,
    "Miasmos" to 4,
    "Nereus" to 4,
    "Hoarder" to 4,
    "Mammoth" to 4,
    "Wreathe" to 4,
    "Primae" to 4,
    "Executioner" to 5,
    "Magnate" to 5,
    "Tormentor" to 5,
    "Condor" to 5,
    "Heron" to 5,
    "Kestrel" to 5,
    "Merlin" to 5,
    "Atron" to 5,
    "Imicus" to 5,
    "Incursus" to 5,
    "Tristan" to 5,
    "Breacher" to 5,
    "Probe" to 5,
    "Rifter" to 5,
    "Slasher" to 5,
    "Venture" to 5,
    "Metamorphosis" to 5,
    "Coercer" to 5,
    "Dragoon" to 5,
    "Corax" to 5,
    "Cormorant" to 5,
    "Algos" to 5,
    "Catalyst" to 5,
    "Talwar" to 5,
    "Thrasher" to 5,
    "Sunesis" to 5,
    "Magnate Navy Issue" to 8,
    "Heron Navy Issue" to 7,
    "Caldari Navy Hookbill" to 6,
    "Griffin Navy Issue" to 6,
    "Crusader" to 7,
    "Malediction" to 7,
    "Crow" to 7,
    "Raptor" to 7,
    "Ares" to 7,
    "Taranis" to 7,
    "Claw" to 7,
    "Stiletto" to 7,
    "Purifier" to 6,
    "Manticore" to 6,
    "Nemesis" to 6,
    "Hound" to 6,
    "Inquisitor" to 5,
    "Bantam" to 5,
    "Navitas" to 5,
    "Burst" to 5,
    "Punisher" to 7,
    "Dramiel" to 7,
    "Garmur" to 7,
    "Succubus" to 7,
    "Daredevil" to 7,
    "Astero" to 7,
    "Skybreaker" to 7,
    "Griffin" to 8,
    "Vigil" to 8,
    "Retribution" to 8,
    "Harpy" to 8,
    "Hawk" to 8,
    "Enyo" to 8,
    "Ishkur" to 8,
    "Jaguar" to 9,
    "Wolf" to 8,
    "Crucifier Navy Issue" to 7,
    "Imperial Navy Slicer" to 6,
    "Imicus Navy Issue" to 6,
    "Federation Navy Comet" to 6,
    "Maulus Navy Issue" to 6,
    "Probe Fleet Issue" to 6,
    "Republic Fleet Firetail" to 6,
    "Vigil Fleet Issue" to 8,
    "Cruor" to 11,
    "Heretic" to 9,
    "Flycatcher" to 9,
    "Eris" to 9,
    "Sabre" to 9,
    "Omen" to 11,
    "Moa" to 11,
    "Thorax" to 11,
    "Rupture" to 11,
    "Stabber" to 11,
    "Coercer Navy Issue" to 8,
    "Cormorant Navy Issue" to 8,
    "Catalyst Navy Issue" to 8,
    "Thrasher Fleet Issue" to 8,
    "Mekubal" to 8,
    "Mamba" to 10,
    "Damavik" to 6,
    "Worm" to 8,
    "Bellicose" to 15,
    "Maulus" to 13,
    "Vengeance" to 10,
    "Nergal" to 11,
    "Caracal" to 11,
    "Confessor" to 14,
    "Hecate" to 14,
    "Svipul" to 14,
    "Crucifier" to 12,
    "Maller" to 13,
    "Vexor" to 13,
    "Kikimora" to 8,
    "Jackdaw" to 15,
    "Pontifex" to 16,
    "Stork" to 16,
    "Magus" to 16,
    "Bifrost" to 16,
    "Kitsune" to 16,
    "Hyena" to 18,
    "Deacon" to 16,
    "Kirin" to 16,
    "Thalia" to 16,
    "Scalpel" to 16,
    "Stormbringer" to 15,
    "Sentinel" to 20,
    "Keres" to 21,
    "Draugur" to 16,
    "Arbitrator" to 21,
    "Celestis" to 19,
    "Omen Navy Issue" to 21,
    "Caracal Navy Issue" to 21,
    "Osprey Navy Issue" to 21,
    "Vexor Navy Issue" to 20,
    "Scythe Fleet Issue" to 20,
    "Stabber Fleet Issue" to 19,
    "Cynabal" to 21,
    "Blackbird" to 20,
    "Augoror Navy Issue" to 23,
    "Vedmak" to 23,
    "Exequror Navy Issue" to 21,
    "Ashimmu" to 30,
    "Orthrus" to 28,
    "Phantasm" to 25,
    "Vigilant" to 25,
    "Stratios" to 25,
    "Gila" to 28,
    "Devoter" to 27,
    "Onyx" to 29,
    "Phobos" to 27,
    "Broadsword" to 29,
    "Augoror" to 25,
    "Osprey" to 25,
    "Exequror" to 25,
    "Scythe" to 23,
    "Oracle" to 31,
    "Naga" to 29,
    "Talos" to 29,
    "Tornado" to 29,
    "Sacrilege" to 28,
    "Zealot" to 30,
    "Cerberus" to 30,
    "Eagle" to 27,
    "Deimos" to 27,
    "Ishtar" to 33,
    "Muninn" to 27,
    "Vagabond" to 30,
    "Falcon" to 35,
    "Huginn" to 37,
    "Rapier" to 37,
    "Rodiva" to 30,
    "Harbinger" to 36,
    "Prophecy" to 34,
    "Drake" to 34,
    "Ferox" to 34,
    "Brutix" to 34,
    "Myrmidon" to 34,
    "Cyclone" to 34,
    "Hurricane" to 34,
    "Gnosis" to 34,
    "Ikitursa" to 33,
    "Pilgrim" to 37,
    "Rook" to 37,
    "Arazu" to 33,
    "Lachesis" to 35,
    "Legion" to 37,
    "Tengu" to 37,
    "Proteus" to 37,
    "Curse" to 43,
    "Loki" to 45,
    "Harbinger Navy Issue" to 44,
    "Prophecy Navy Issue" to 44,
    "Drake Navy Issue" to 44,
    "Ferox Navy Issue" to 44,
    "Brutix Navy Issue" to 44,
    "Myrmidon Navy Issue" to 42,
    "Hurricane Fleet Issue" to 42,
    "Cyclone Fleet Issue" to 44,
    "Drake Navy Issue" to 44,
    "Ferox Navy Issue" to 44,
    "Khizriel" to 46,
    "Guardian" to 45,
    "Basilisk" to 45,
    "Oneiros" to 45,
    "Scimitar" to 45,
    "Drekavac" to 44,
    "Alligator" to 46,
    "Thunderchild" to 44,
    "Absolution" to 49,
    "Damnation" to 46,
    "Nighthawk" to 49,
    "Vulture" to 46,
    "Astarte" to 49,
    "Claymore" to 49,
    "Sleipnir" to 49,
    "Apocalypse" to 55,
    "Raven" to 53,
    "Rokh" to 51,
    "Dominix" to 53,
    "Megathron" to 51,
    "Maelstrom" to 48,
    "Tempest" to 51,
    "Typhoon" to 53,
    "Praxis" to 51,
    "Zarmazd" to 55,
    "Abaddon" to 55,
    "Armageddon" to 55,
    "Scorpion" to 55,
    "Hyperion" to 55,
    "Eos" to 52,
    "Redeemer" to 61,
    "Sin" to 60,
    "Panther" to 55,
    "Apocalypse Navy Issue" to 65,
    "Armageddon Navy Issue" to 65,
    "Raven Navy Issue" to 63,
    "Scorpion Navy Issue" to 61,
    "Dominix Navy Issue" to 63,
    "Megathron Navy Issue" to 61,
    "Tempest Fleet Issue" to 61,
    "Typhoon Fleet Issue" to 63,
    "Paladin" to 63,
    "Golem" to 61,
    "Kronos" to 59,
    "Vargur" to 63,
    "Machariel" to 62,
    "Widow" to 70,
    "Bhaalgorn" to 68,
    "Rattlesnake" to 65,
    "Barghest" to 71,
    "Nightmare" to 62,
    "Vindicator" to 68,
    "Leshak" to 71,
    "Babaroga" to 65,
    "Outrider" to 16,
    "Tholos" to 16,
    "Deluge" to 12,
    "Squall" to 10,
)


/**
 * The composition rules for Anger Games 7.
 */
private class Ag7CompositionRules(eveData: EveData): AngerGamesCompositionRules(
    eveData = eveData,
    maxCompositionSize = 7,
    maxCompositionCost = 200,
    shipCostByName = ShipCostByName
) {

    /**
     * Returns the duplicate penalty for the given ship type.
     */
    private fun penaltyForDuplicate(shipType: ShipType): Int {
        with(eveData) {
            return when (shipType.sizeClass) {
                ShipSizeClass.CORVETTE -> 0
                ShipSizeClass.INDUSTRIAL -> 2
                ShipSizeClass.FRIGATE -> 0
                ShipSizeClass.LOGI_FRIGATE -> 1
                ShipSizeClass.DESTROYER -> 1
                ShipSizeClass.CRUISER -> 2
                ShipSizeClass.LOGI_CRUISER -> 2
                ShipSizeClass.BATTLECRUISER -> 3
                ShipSizeClass.BATTLESHIP -> 4
                ShipSizeClass.OTHER -> 0
            }
        }
    }


    override fun shipsCost(shipTypes: List<ShipType?>): List<Int> {
        return shipsCostWithDuplicatePenalty(
            shipTypes = shipTypes,
            shipCostByType = shipCostByType,
            penalty = ::penaltyForDuplicate
        )
    }


    override fun marginalShipCostComputation(usedShipTypes: List<ShipType?>): MarginalShipCostComputation {
        return marginalShipCostComputationWithDuplicatePenalty(
            usedShipTypes = usedShipTypes,
            shipCostByType = shipCostByType,
            penalty = ::penaltyForDuplicate
        )
    }


}


/**
 * The fitting rules for Anger Games 7.
 */
private class Ag7FittingRules(
    eveData: EveData
): AngerGamesFittingRules(
    eveData = eveData,
    svipulAndConfessorCanBeLogistics = false
) {

    override fun isModuleLegal(
        moduleType: ModuleType,
        shipType: ShipType,
        isFlagship: Boolean,
    ): Boolean {
        if (!super.isModuleLegal(moduleType, shipType, isFlagship))
            return false

        with(eveData) {
            // The Zarmazd can only fit the mutadaptive armor repairer
            val isRegularArmorRepairer = moduleType.isRemoteArmorRepairer(includeAncillary = true, includeMutadaptive = false)
            if (isRegularArmorRepairer && (shipType.name == "Zarmazd"))
                return false
        }

        return true
    }
}


/**
 * The rules for Anger Games 7.
 */
private fun angerGames7Rules(eveData: EveData) =
    TournamentRules(
        compositionRules = Ag7CompositionRules(eveData),
        fittingRules = Ag7FittingRules(eveData)
    )
