package theorycrafter.utils

import androidx.compose.foundation.border
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.LocalContentColor
import androidx.compose.runtime.Composable
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.draw.drawBehind
import androidx.compose.ui.graphics.Brush
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.Shadow
import androidx.compose.ui.graphics.drawscope.DrawScope
import androidx.compose.ui.text.AnnotatedString
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import compose.widgets.SingleLineText
import theorycrafter.ui.TheorycrafterTheme


/**
 * A box that draws background and text with shadow.
 *
 * This version takes a [drawBackground] function that draws the background.
 */
@Composable
fun ColoredTextBox(
    text: String,
    drawBackground: DrawScope.() -> Unit,
    textColor: Color,
    textShadowColor: Color,
    textModifier: Modifier = Modifier,
    modifier: Modifier = Modifier
) {
    Box(
        contentAlignment = Alignment.Center,
        modifier = modifier
            .border(
                width = Dp.Hairline,
                color = LocalContentColor.current.copy(alpha = 0.25f),
                shape = RoundedCornerShape(2.dp)
            )
            .clip(
                RoundedCornerShape(2.dp)
            )
            .drawBehind(drawBackground)
    ) {
        SingleLineText(
            text = text.withShadow(textShadowColor),
            overflow = TextOverflow.Visible,
            textAlign = TextAlign.Center,
            style = TheorycrafterTheme.textStyles.percentageView,
            color = textColor,
            modifier = textModifier
        )
    }
}


/**
 * A box that draws background and text with shadow.
 *
 * This version draws a solid background.
 */
@Composable
fun ColoredTextBox(
    text: String,
    background: Color,
    textColor: Color,
    textShadowColor: Color,
    textModifier: Modifier = Modifier,
    modifier: Modifier = Modifier
) {
    val brush = rememberColoredTextBoxBackgroundBrush(background)
    ColoredTextBox(
        text = text,
        drawBackground = { drawRect(brush) },
        textColor = textColor,
        textShadowColor = textShadowColor,
        textModifier = textModifier,
        modifier = modifier
    )
}


/**
 * The brush for drawing the filled part of the perecentage view.
 */
@Composable
fun rememberColoredTextBoxBackgroundBrush(color: Color): Brush {
    return remember(color) {
        Brush.linearGradient(
            listOf(color.darker(0.075f), color.lighter(0.075f))
        )
    }
}


/**
 * Returns an [AnnotatedString] of the given string with a shadow of the given color.
 */
private fun String.withShadow(shadowColor: Color) =
    AnnotatedString(
        text = this,
        spanStyle = SpanStyle(
            shadow = Shadow(
                color = shadowColor,
                blurRadius = 1.5f
            )
        )
    )